Grogol (600 pts)

Overview
We are given a 64-bit ELF binary acting as a custom stack-based calculator. TL;DR: It tokenizes (strtok) the input, hashes the tokens to be used as opcodes, and executes them on a custom Virtual Machine (VM). The goal is to provide an input string that results in the final stack’s top value of 31337.
Analysis
Upon opening the binary in IDA Pro, we analyze main. From here on, decompilation code snippets have been renamed for readability; for you. Interestingly, this is a RISC-V binary!! Something refreshing for once… The program sets up standard buffering and reads input via Fgets.
if ( !Fgets(Input, 4096, STDIN) ) goto L_ExitFail;
if ( (unsigned __int64)Strlen(Input) > 21 ) {
  Print(2, "Your input is too long! ->>>>");
  // ...
}

See that?! We have a constraint. The input length must be  less than or equal to 21 characters. Next…
The input is split into tokens using Strtok (space, tab, newline). These tokens are then hashed using a custom algo, then mapped to opcodes with actions such as addition, multiplication, and push-to-stack. It calculates a rolling XOR hash for each token; effectively, we can just brute-force all possible values for a byte (256 of them) and create a reverse map of the hashes opcodes.
Let's reconstruct the hashing algorithm from the decompiled C code:
state_A = 1;
state_B = 0;
if ( *pBuf1024 )
{
  pBuf1024_3 = pBuf1024;
  LowNib = 1;
  Len_1 = 1LL - (_QWORD)pBuf1024;
  do
  {
    Len = Len_1 + (_BYTE)pBuf1024_3; // decompile error; this should be str len
    state_A = LowNib >> 1;
    state_B = (__int64)&pBuf1024_3[Len_1];
    Val = (unsigned __int8)*pBuf1024_3++;
    LowNib = Val ^ (LowNib >> 1);
    state_A = LowNib;
  }
  while ( *pBuf1024_3 );
  if ( ((unsigned __int8)(16 * (*pBuf1024 ^ Len)) | (unsigned __int64)(LowNib & 0xF)) == 0x27 )
  {
    A_ccumulator += 30;
    ::A_ccumulator = A_ccumulator;
  }
}

This loop, which reoccurs for every time an opcode check is made (talk about redundancy), calculates the State values, and then does a check to see if it matches the opcode (0x27 in the example). The multiply by 16 is the same as << 4, meaning, it shifts the value into the higher nibble.
We do not need to reverse the collision check perfectly if we simply brute-force short strings (a char, even) that result in the opcodes.
As I said, the binary implements a custom stack machine:
Accumulator: stores the current intermediate value.
HighPart: Stores large multiplied values (e.g., when multiplying by 1000 or 1,000,000).
Multiplier: Toggles between 1 and -1 (for subtraction/negative numbers).
Stack: Used to store operands and operators.
  else
  {
    Idx = 0;
  }
  Buf1024[Idx] = 0;
  pBuf1024 = (_BYTE *)Buf1024[0]; // input
  Multiplier = 1; // sign (negative or positive)
  A_ccumulator = ::A_ccumulator; // technically LowPart
  B_HighPart = ::B_HighPart;
  C_ExprStackPtr = ::C_ExprStackPtr;
  D = ::D;

The VM processes these tokens in a loop. When the loop finishes, it performs a final calculation.
// around lines 855-859
v210 = ::A_ccumulator * Multiplier;
// ...
v212[1] = v210 + v209; // Adds Accumulator + HighPart

It then checks if the top of the stack is 31337.
if ( STACK[2 * STACK_TOP + 1] != 31337 )

Next, we map the opcodes to the actions they perform. I used Helix to make things easier for me, and I got the following:
op | action
6A | += 7
34 | += 40
DF | += 18
1f | += 15
a5 | += 9
79 | += 2
28 | += 5
71 | += 10
38 | += 11
6F | += 19
2D | += 12
EE | += 14
00 | += 6
2F | += 4
CE | += 13
31 | += 50
82 | += 90
AF | += 17
02 | += 8
11 | += 3
67 | += 60
41 | += 70
27 | += 30
c9 | += 1
4E | += 16
30 | += 80
20 | += 20
f2 | *= 100
C0 | *= Multiplier * 1e3  -> B
aa | *= Multiplier * 1e6  -> B
5a | *= Multiplier * 1e9  -> B
CA | *= Multiplier * 1e12 -> B
6e | Multiplier = -1
10 | push imm; add multiplication operator
37 | push imm; add division operator
89 | push imm; add subtraction operator
4c | push imm; add addition operator
e8 | push imm; add modulo operator

For the push instructions, they push the current immediate value, and also an operator.
Solution
Now that we know what we’re dealing with, let’s find the opcodes to construct 31337. We are limited by input length (21 chars), so we cannot simply add 1 thirty-one thousand times. We must use multiplication.
31337 = 31000 + 337
31337 = (31 x 1000) + (3 x 100) + 37
31337 = ((30 + 1) x 1000) + (3 x 100) + 30 + 7
So, the VM Instruction Sequence is
Load 30: ADD_30 (Accumulator = 30)
Add 1: ADD_1 (Accumulator = 31)
Multiply by 1000 & Store: MUL_1000
B_HighPart becomes 31,000.
A_ccumulator becomes 0.
Load 3: ADD_3 (Accumulator = 3)
Multiply by 100: MUL_100 (Accumulator = 300)
Note: MUL_100 acts directly on A_ccumulator, unlike MUL_1000.
Add 30: ADD_30 (Accumulator = 330)
Add 7: ADD_7 (Accumulator = 337)
At program termination, the VM sums A_ccumulator and B_HighPart:
337 + 31000 = 31337
What’s left is getting the input that translates to our target opcodes. Since we can't type "ADD_30", we need to find short strings that generate the required hash opcodes. I tried with just a char, but none is the opcode we want. We’ll use itertools.product to brute-force these collisions. Referring back to the opcodes we extracted above, our payload will be:
    ADD_30: 0x27
    ADD_1: 0xC9
    MUL_1000: 0xC0
    ADD_3: 0x11
    MUL_100: 0xF2
    ADD_7: 0x6A
solve.py
import string
import itertools
from pwn import *

# === Configuration ===
HOST = '129.226.145.210'
PORT = 317

# === Hashing Logic (Reversed from C) ===
def calculate_hash(s):
    v13 = 1
    for char in s:
        c = ord(char)
        v13 = c ^ (v13 >> 1) # The rolling XOR

    first = ord(s[0])
    v15 = len(s)

    # The check found in the 'if' statements:
    # ((unsigned __int8)(16 * (*pBuf1024 ^ v18)) | (unsigned __int64)(State & 0xF))
    # v18 in C was calculated using pointer math, effectively just the length for small strings
    
    term1 = ((first ^ v15) & 0xF) << 4
    term2 = v13 & 0xF
    return term1 | term2

def find_collision(target_hash):
    # Brute force 2 char strings (efficient and keeps payload short)
    chars = string.ascii_letters + string.digits
    for s in itertools.product(chars, repeat=2):
        word = "".join(s)
        if calculate_hash(word) == target_hash:
            return word
    # Fallback to 1 char if needed
    for s in chars:
        if calculate_hash(s) == target_hash:
            return s
    return None

# === Payload Construction ===
# Logic: ((30 + 1) * 1000) + (3 * 100) + 30 + 7 = 31337
needed_hashes = {
    'ADD_30': 0x27,
    'ADD_1':  0xC9,
    'MUL_1k': 0xC0,
    'ADD_3':  0x11,
    'MUL_100':0xF2,
    'ADD_7':  0x6A
}

print("[*] Finding collisions...")
word_map = {}
for name, h in needed_hashes.items():
    word = find_collision(h)
    if word:
        print(f"    {name:<8} (0x{h:02X}) -> {word}")
        word_map[name] = word
    else:
        log.error(f"Could not find collision for {name}")

# Construct the sequence
payload_tokens = [
    word_map['ADD_30'],  # 30
    word_map['ADD_1'],   # 31
    word_map['MUL_1k'],  # Move 31000 to HighPart, Acc=0
    word_map['ADD_3'],   # 3
    word_map['MUL_100'], # 300
    word_map['ADD_30'],  # 330
    word_map['ADD_7']    # 337
]

payload = " ".join(payload_tokens)
print(f"[*] Payload length: {len(payload)}")
print(f"[*] Payload: {payload}")

# === Exploitation ===
r = remote(HOST, PORT)
r.recvuntil(b'Input > ')
r.sendline(payload.encode())

# Interaction to grab flag
r.interactive()



Flag: cyberwave{belajar_rev_di_google_aja}

