#!/usr/bin/env python3
import sys

def solve():
    # The ciphertext array extracted from the HTML
    seq = [
        697438994, 112812148, 1601952528, 848567091, 1409540622, 719284623,
        4183535244, 2852213902, 2109397207, 3611470734, 604567242, 1692610300,
        2414225221, 1979551723, 3174382114, 425190723, 1060279654, 4283219352,
        1099139615, 3427871953, 2056419824, 103242998, 2789231820, 97749902,
        3832000502, 2437931514, 337329827, 1784389836, 1971115025, 2430188600,
        420768160, 2890064672, 3927353914, 2033350795, 372941141, 2974609317,
        1442092933, 2838369745, 3198352587, 230640255, 2641503210, 1721308159,
        3764390994
    ]

    # --- LCG Constants ---
    M = 2147483647
    A = 48271
    # Calculate Modular Inverse for reversing the RNG steps
    # Using Fermat's Little Theorem: A^(M-2) % M is the inverse
    A_inv = pow(A, M - 2, M) 

    print("[*] Starting Seed Recovery...")

    # Extract leaked lower 24 bits
    leak_0 = seq[0] & 0xFFFFFF  # Corresponds to 3rd RNG output (r3)
    leak_1 = seq[1] & 0xFFFFFF  # Corresponds to 6th RNG output (r6)
    
    seed = None
 
    # Brute-force the upper 7 bits (approx 128 possibilities)
    # We are reconstructing r3
    for k in range(128): 
        candidate_r3 = leak_0 + (k << 24)
        
        # Candidate must be within the modulus
        if candidate_r3 >= M:
            break
        
        # Step BACKWARDS to find the initial seed (s0)
        # s0 -> r1 -> r2 -> r3 (candidate)
        r2 = (candidate_r3 * A_inv) % M
        r1 = (r2 * A_inv) % M
        s0 = (r1 * A_inv) % M

        # Step FORWARDS to verify against leak_1 (r6)
        # r3 -> r4 -> r5 -> r6
        r4 = (candidate_r3 * A) % M
        r5 = (r4 * A) % M
        r6 = (r5 * A) % M
        
        if (r6 & 0xFFFFFF) == leak_1:
            seed = s0
            print(f"[+] Found Correct Seed: {seed}")
            break
            
    if seed is None:
        print("[-] Failed to find seed. Logic error suspected.")
        return

    # --- Decryption Phase ---
    current_seed = seed
    
    # Internal RNG helper
    def rng():
        nonlocal current_seed
        current_seed = (current_seed * A) % M
        return current_seed

    def nibble_swap(x):
        return ((x & 0x0F) << 4) | ((x & 0xF0) >> 4)

    flag = ""
    print("[*] Decrypting sequence...")

    for val in seq:
        # Replicate the 3 RNG calls per char
        r1 = rng() # Used for adj
        r2 = rng() # Used for XOR
        r3 = rng() # Used for Padding (which we saw in the leak)

        # 1. Reverse XOR
        encrypted_byte = (val >> 24) & 0xFF
        xor_key = r2 & 0xFF
        step_1 = encrypted_byte ^ xor_key

        # 2. Reverse Nibble Swap (Symmetric operation)
        step_2 = nibble_swap(step_1)

        # 3. Reverse Adjustment
        # Original: x = (x + adj) % 256
        # Inverse:  x = (x - adj) % 256
        adj = (r1 % 93) + 33
        plain_char = (step_2 - adj) % 256

        flag += chr(plain_char)

    print(f"\n[SUCCESS] Flag: {flag}")

if __name__ == "__main__":
    solve()
